<?php

namespace TdyGames\Controller\Rest;

use TdyGames\Model\Attribute;
use TdyGames\InputFilter\Attribute as AttributeInputFilter;
use TdyCommons\Controller\BaseRestController;
use Zend\View\Model\JsonModel;

class AttributesController extends BaseRestController
{
    protected $identifierName = 'id';

    /**
     * @return JsonModel
     */
    public function getList()
    {
        $this->getLogger()->debug('Games\AttributesController::getList() start.');

        $q = $this->params()->fromQuery('q', '');
        $p = (int) $this->params()->fromQuery('p', 1);
        $l = (int) $this->params()->fromQuery('l', $this->getSettingsTable()->get('items-per-page'));
        $s = $this->params()->fromQuery('s', []);
        $f = $this->params()->fromQuery('f', []);
        $x = [];

        $isPaginated = $l > 0;

        $resultSet = $this->getGameAttributesTable()->setIndent(self::LOGGER_INDENT)->fetch($q, $x, $f, $s, $isPaginated);
        $resultSet->setCurrentPageNumber((int) $p);
        $resultSet->setItemCountPerPage($l);

        if ($p > 0) {
            $i = (($p - 1) * $l);
        } else {
            $i = 0;
        }

        $attributes = iterator_to_array($resultSet, false);

        /**
         * @var int       $k
         * @var Attribute $attribute
         */
        foreach ($attributes as $k => $attribute) {
            $attribute->setServiceLocator($this->getServiceLocator());
            $attributes[$k]->row = ++$i;
            $attributes[$k]      = $attribute->toStdClass();
        }

        $this->getLogger()->debug('Games\AttributesController::getList() end.');

        return $this->statusOk([
            'entries'    => $attributes,
            'pagination' => [
                'pageNumber' => $p,
                'pageSize'   => $l,
                'totalItems' => $resultSet->getTotalItemCount(),
                'totalPages' => $resultSet->count(),
            ],

        ]);
    }

    /**
     * @param int|string $usr
     *
     * @return JsonModel
     */
    public function get($usr)
    {
        $attribute = $this->getGameAttributesTable()->get($usr);

        return $this->statusOk([
            'entry'      => $attribute->toStdClass(),
            'pagination' => [
                'pageNumber' => 1,
                'pageSize'   => 1,
                'totalItems' => 1,
                'totalPages' => 1,
            ],
        ]);
    }

    /**
     * @param mixed $data
     *
     * @return JsonModel
     */
    public function create($data)
    {
        $this->getLogger()->debug('Games\AttributesController::create() start.');

        $data['id'] = 0;

        $inputFilter = new AttributeInputFilter($this->getServiceLocator());
        $inputFilter->setData($data);

        if ($inputFilter->isValid()) {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is valid. Continue.');

            $data = $inputFilter->getValues();

            $attribute            = new Attribute($data);
            $attribute->createdBy = $this->getCurrentAdmin()->id;

            $attribute = $this->getGameAttributesTable()->setIndent(self::LOGGER_INDENT)->save($attribute);

            $this->getLogger()->info(self::LOGGER_INDENT . 'Attribute ' . $attribute->name . ' has been created.');
            $this->getLogger()->debug('Games\AttributesController::create() end.');

            return $this->statusOk([
                'entry'      => $attribute->toStdClass(),
                'pagination' => [
                    'pageNumber' => 1,
                    'pageSize'   => 1,
                    'totalItems' => 1,
                    'totalPages' => 1,
                ],
            ]);
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is invalid. Bugging out.');
            $this->getLogger()->debug('Games\AttributesController::create() end.');

            return $this->statusBadRequest([
                'system'    => [
                    'id'          => 4001,
                    'description' => 'Invalid Request'
                ],
                'companies' => [
                    'description' => 'Submission is invalid.',
                    'details'     => $inputFilter->getMessages()
                ]
            ]);
        }
    }

    /**
     * @param int|string $id
     * @param mixed      $data
     *
     * @return JsonModel
     */
    public function update($id, $data)
    {
        $this->getLogger()->debug('Games\AttributesController::update() start.');

        $data['id'] = (int) $id;

        $inputFilter = new AttributeInputFilter($this->getServiceLocator(), true);
        $inputFilter->setData($data);

        if ($inputFilter->isValid()) {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is valid. Continue.');

            $attribute            = new Attribute($data);
            $attribute->updatedBy = $this->getCurrentAdmin()->id;
            $attribute            = $this->getGameAttributesTable()->setIndent(self::LOGGER_INDENT)->save($attribute);

            $this->getLogger()->info(self::LOGGER_INDENT . 'Attribute ' . $attribute->name . ' has been updated.');
            $this->getLogger()->debug('Games\AttributesController::update() end.');

            return $this->statusOk([
                'entry'      => $attribute->toStdClass(),
                'pagination' => [
                    'pageNumber' => 1,
                    'pageSize'   => 1,
                    'totalItems' => 1,
                    'totalPages' => 1,
                ],
            ]);
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is invalid. Will not update.');
            $this->getLogger()->debug('Games\AttributesController::update() end.');

            return $this->statusBadRequest([
                'system'    => [
                    'id'          => 4001,
                    'description' => 'Invalid Request'
                ],
                'companies' => [
                    'description' => 'Submission is invalid.',
                    'details'     => $inputFilter->getMessages(),
                ]
            ]);
        }
    }

    /**
     * @param int|string $id
     *
     * @return JsonModel
     * @throws \Exception
     */
    public function delete($id)
    {
        $this->getLogger()->debug('Games\AttributesController::delete() start.');

        $this->getLogger()->debug(self::LOGGER_INDENT . 'Getting info.');

        if ($id > 0) {
            $attribute = $this->getGameAttributesTable()->get($id);
            if ($attribute) {

                $this->getLogger()->debug(self::LOGGER_INDENT . 'Attribute ID  : ' . $id);
                $this->getLogger()->debug(self::LOGGER_INDENT . 'Attribute Name: ' . $attribute->name);

                $attribute->deletedBy = $this->getCurrentAdmin()->id;
                $this->getGameAttributesTable()->setIndent(self::LOGGER_INDENT)->delete($attribute);

                $this->getLogger()->info(self::LOGGER_INDENT . 'Attribute ' . $attribute->name . ' has been deleted.');
                $this->getLogger()->debug('Games\AttributesController::delete() end.');

                return $this->statusOk([
                    'entry'      => null,
                    'pagination' => [
                        'pageNumber' => 1,
                        'pageSize'   => 1,
                        'totalItems' => 1,
                        'totalPages' => 1,
                    ],
                ]);
            } else {
                $this->getLogger()->debug(self::LOGGER_INDENT . 'Attribute ID ' . $id . ' does not exists.');
                $this->getLogger()->debug('Games\AttributesController::delete() end.');

                return $this->statusNotFound();
            }
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Attribute ID is requred for deletion.');
            $this->getLogger()->debug('Games\AttributesController::delete() end.');

            return $this->statusNotFound();
        }
    }

}
