<?php

namespace TdyCommons\Table;

use TdyCommons\Model;
use Zend\Db\ResultSet\HydratingResultSet;
use Zend\Db\Sql\Sql;
use Zend\Db\Sql\Predicate;
use Zend\Db\Sql\Select;
use Zend\Db\Sql\Where;
use Zend\Paginator\Adapter\DbSelect;
use Zend\Paginator\Paginator;

/**
 * Option table class.
 *
 * @author James Lloyd Atwil <james@nadows.com>
 */
class Options extends Table
{

    /**
     * @param int    $c
     * @param string $q
     * @param array  $sort
     * @param bool   $paginated
     *
     * @return bool|HydratingResultSet|Paginator
     */
    public function fetch($c, $q = '', $sort = array(), $paginated = false)
    {
        $select     = new Select(TBL_COMMONS_SELECTIONS_OPTIONS);
        $conditions = array();

        $where = new Where(array());
        $where->addPredicate(new Predicate\Expression('selection_id = ?', $c));
        $where->addPredicate(new Predicate\Expression('is_active = 1'));
        $conditions[] = $where;

        if (!empty($q)) {
            $where = new Where(array(), Predicate\PredicateSet::OP_OR);
            $where->addPredicate(new Predicate\Like('name', '%' . $q . '%'));
            $conditions[] = $where;
        }

        $select->where($conditions, Predicate\PredicateSet::OP_AND);

        if (empty($sort)) {
            $select->order('name ASC');
        } else {
            $select->order($sort);
        }

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\Option());

        if ($paginated) {
            $paginatorAdapter = new DbSelect($select, $this->tableGateway->getAdapter(), $resultSet);
            $paginator        = new Paginator($paginatorAdapter);

            return $paginator;
        } else {
            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($select);
            $results   = $statement->execute();

            $resultSet->initialize($results);

            if (!$resultSet) {
                return false;
            } else {
                return $resultSet;
            }
        }
    }

    /**
     * @param $id
     *
     * @return Model\Option|bool
     */
    public function get($id)
    {
        $rowset = $this->tableGateway->select(array('id' => $id));
        $row    = $rowset->current();
        if (!$row) {
            return false;
        }

        return $row;
    }

    public function getByOptionValue($selection, $value)
    {
        $rowset = $this->tableGateway->select(array(
            'selection_id' => $selection,
            'value'        => $value
        ));
        $row    = $rowset->current();
        if (!$row) {
            return false;
        }

        return $row;
    }

    /**
     * Save the Option to the table.
     *
     * @param Model\Option $option
     *
     * @return Model\Option
     * @throws \Exception
     */
    public function save(Model\Option $option)
    {
        $data = array(
            'selection_id' => $option->selectionId,
            'slug'         => $option->slug,
            'name'         => $option->name,
            'value'        => $option->value,
            'ehris_id'     => $option->ehrisId,
            'ovid'         => $option->ovid,
            'gl_code'      => $option->glCode
        );

        $id = $option->id;
        if ($id === 0) {
            $data['is_active']  = 1;
            $data['created_by'] = $option->createdBy;
            $data['created_on'] = date('Y-m-d H:i:s');
            $this->tableGateway->insert($data);
            $id = $this->tableGateway->getLastInsertValue();

            return $this->get($id);
        } else {
            if ($this->get($option->id)) {
                $data['updated_by'] = $option->updatedBy;
                $data['updated_on'] = date('Y-m-d H:i:s');
                $this->tableGateway->update($data, array('id' => $option->id));

                return $this->get($id);
            } else {
                throw new \Exception('Option ID does not exist');
            }
        }
    }

    /**
     * Delete the Option from the table. Actually, it simply sets the Option to inactive.
     *
     * @param Model\Option $option
     *
     * @throws \Exception
     */
    public function delete(Model\Option $option)
    {
        if ($this->get($option->id)) {
            $data = array(
                'is_active'  => 0,
                'deleted_by' => $option->deletedBy,
                'deleted_on' => date('Y-m-d H:i:s'),
            );
            $this->tableGateway->update($data, array('id' => $option->id));
        } else {
            throw new \Exception('Option ID does not exist');
        }
    }

    /**
     * Sets the Option to active.
     *
     * @param Model\Option $option
     *
     * @throws \Exception
     */
    public function restore(Model\Option $option)
    {
        if ($this->get($option->id)) {
            $data = array(
                'is_active'  => 1,
                'updated_by' => $option->updatedBy,
                'updated_on' => date('Y-m-d H:i:s'),
            );
            $this->tableGateway->update($data, array('id' => $option->id));
        } else {
            throw new \Exception('Option ID does not exist');
        }
    }

    /**
     * @param Model\Option $option
     */
    public function burn(Model\Option $option)
    {
        $this->tableGateway->delete(array('id' => $option->id));
    }

}
