<?php

namespace TdySports\Table;

use TdySports\Model;
use TdyCommons\Table\Table;
use Zend\Db\ResultSet\HydratingResultSet;
use Zend\Db\Sql\Delete;
use Zend\Db\Sql\Insert;
use Zend\Db\Sql\Predicate;
use Zend\Db\Sql\Select;
use Zend\Db\Sql\Sql;
use Zend\Db\Sql\Update;
use Zend\Db\Sql\Where;
use Zend\Paginator\Adapter\DbSelect;
use Zend\Paginator\Paginator;

/**
 * Class Teams
 *
 * @package TdySports\Table
 *
 * @author  James Lloyd Atwil <james@nadows.com>
 */
class Teams extends Table
{

    /**
     * @param string $q
     * @param array  $exclusion
     * @param array  $filters
     * @param array  $sort
     * @param bool   $paginated
     *
     * @return bool|HydratingResultSet|Paginator
     */
    public function fetch($q = '', $exclusion = [], $filters = [], $sort = [], $paginated = false)
    {
        $this->getLogger()->debug($this->indent . 'Teams::fetch() start.');

        $select = new Select();
        $select->from($this->getTableGateway()->getTable());
        $conditions = [];

        $where = new Where([]);
        $where->addPredicate(new Predicate\Expression('is_active = ?', 1));
        $conditions[] = $where;

        if (!empty($q)) {
            $where = new Where([], Predicate\PredicateSet::OP_OR);
            $where->addPredicate(new Predicate\Like($this->columns['slug'], '%' . $q . '%'));
            $where->addPredicate(new Predicate\Like($this->columns['name'], '%' . $q . '%'));
            $conditions[] = $where;
        }

        if (!empty($filters)) {
            $where = new Where([]);
            foreach ($filters as $k => $v) {
                if (is_array($v)) {
                    $where->addPredicate(new Predicate\In($k, $v));
                } else {
                    $where->addPredicate(new Predicate\Expression($k . ' = ?', $v));
                }
            }
            $conditions[] = $where;
        }

        if (!empty($exclusion)) {
            $where = new Where([]);
            $where->addPredicate(new Predicate\NotIn('id', $exclusion));
            $conditions[] = $where;
        }

        $select->where($conditions, Predicate\PredicateSet::OP_AND);

        if (!is_null($sort)) {
            if ($this->isSortEmpty($sort)) {
                $select->order([
                    $this->columns['ordering'] => 'ASC',
                    $this->columns['name']     => 'ASC',
                ]);
            } else {
                $select->order($sort);
            }
        } else {
            $select->order($sort);
        }

        $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $select->getSqlString($this->getAdapter()->getPlatform()));

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\Team());

        if ($paginated) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: YES');

            $paginatorAdapter = new DbSelect($select, $this->getTableGateway()->getSql(), $resultSet);
            $paginator        = new Paginator($paginatorAdapter);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns Paginator instance.');
            $this->getLogger()->debug($this->indent . 'Teams::fetch() end.');

            return $paginator;
        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: NO');

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($select);

            $results = $statement->execute();
            $resultSet->initialize($results);

            if (!$resultSet) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns FALSE');
                $this->getLogger()->debug($this->indent . 'Teams::fetch() end.');

                return false;
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns HydratingResultSet instance.');
                $this->getLogger()->debug($this->indent . 'Teams::fetch() end.');

                return $resultSet;
            }
        }
    }

    /**
     * Returns the Sport.
     *
     * @param int $id
     *
     * @return Model\Team|null
     * @throws \Exception
     */
    public function get($id)
    {
        $rowset = $this->tableGateway->select([$this->columns['id'] => $id]);
        $row    = $rowset->current();
        if (!$row) {
            return null;
        }

        return $row;
    }

    /**
     * @param $leagueId
     * @param $slug
     *
     * @return Model\Team|null
     */
    public function getBySlug($leagueId, $slug)
    {
        $rowset = $this->tableGateway->select([$this->columns['slug'] => $slug, $this->columns['league-id'] => $leagueId]);
        $row    = $rowset->current();
        if (!$row) {
            return null;
        }

        return $row;
    }

    /**
     * @param $leagueId
     * @param $code
     *
     * @return Model\Team|null
     */
    public function getByCode($leagueId, $code)
    {
        $rowset = $this->tableGateway->select([$this->columns['code'] => $code, $this->columns['league-id'] => $leagueId]);
        $row    = $rowset->current();
        if (!$row) {
            return null;
        }

        return $row;
    }

    /**
     * @param Model\Team $team
     *
     * @return Model\Team
     * @throws \Exception
     */
    public function save(Model\Team $team)
    {
        $this->getLogger()->debug($this->indent . 'Teams::save() start.');

        $data = [
            $this->columns['slug']        => $team->slug,
            $this->columns['sport-id']    => $team->sportId,
            $this->columns['league-id']   => $team->leagueId,
            $this->columns['group-id']   => $team->groupId,
            $this->columns['name']        => $team->name,
            $this->columns['short-name']  => $team->shortName,
            $this->columns['description'] => $team->description,
        ];

        $id = (int) $team->id;
        if ($id == 0) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Inserting Team record.');

            $data['is_active']  = 1;
            $data['created_by'] = $team->createdBy;
            $data['created_on'] = date('Y-m-d H:i:s');

            $insert = new Insert($this->getTableGateway()->getTable());
            $insert->values($data);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $insert->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($insert);

            $statement->execute();

            $id = $this->getAdapter()->getDriver()->getLastGeneratedValue();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ' . $id . ' has been inserted.');
            $this->getLogger()->debug($this->indent . 'Teams::save() end.');

            return $this->get($id);
        } else {
            if ($this->get($id)) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Updating Team record.');

                $data[$this->columns['updated-by']] = $team->updatedBy;
                $data[$this->columns['updated-on']] = date('Y-m-d H:i:s');

                $update = new Update($this->getTableGateway()->getTable());
                $update->set($data);
                $update->where([$this->columns['id'] => $id]);

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

                $sql       = new Sql($this->tableGateway->getAdapter());
                $statement = $sql->prepareStatementForSqlObject($update);

                $statement->execute();

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ' . $id . ' has been updated.');
                $this->getLogger()->debug($this->indent . 'Teams::save() end.');

                return $this->get($id);
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ID ' . $id . ' does not exists. Update failed.');
                $this->getLogger()->debug($this->indent . 'Teams::save() end.');

                throw new \Exception('Sport ID does not exist');
            }
        }
    }

    /**
     * @param Model\Team $team
     *
     * @throws \Exception
     */
    public function delete(Model\Team $team)
    {
        $this->getLogger()->debug($this->indent . 'Teams::delete() start.');

        if ($this->get($team->id)) {
            $data = [
                $this->columns['is-active']  => 0,
                $this->columns['deleted-by'] => $team->deletedBy,
                $this->columns['deleted-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $team->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ' . $team->id . ' has been deleted.');
            $this->getLogger()->debug($this->indent . 'Teams::delete() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ID ' . $team->id . ' does not exists. Deletion failed.');
            $this->getLogger()->debug($this->indent . 'Teams::delete() end.');

            throw new \Exception('Sport ID does not exist');
        }
    }

    /**
     * Restore Sport to the table. Sets the Sport to active.
     *
     * @param Model\Team $team
     *
     * @throws \Exception
     */
    public function restore(Model\Team $team)
    {
        $this->getLogger()->debug($this->indent . 'Teams::restore() start.');

        if ($this->get($team->id)) {
            $data = [
                $this->columns['is-active']  => 1,
                $this->columns['updated-by'] => $team->updatedBy,
                $this->columns['updated-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $team->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ' . $team->id . ' has been restored.');
            $this->getLogger()->debug($this->indent . 'Teams::restore() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ID ' . $team->id . ' does not exists. Restoration failed.');
            $this->getLogger()->debug($this->indent . 'Teams::restore() end.');

            throw new \Exception('Team ID does not exist');
        }
    }

    /**
     * Permanently delete the Team.
     *
     * @param Model\Team $team
     *
     * @throws \Exception
     */
    public function purge(Model\Team $team)
    {
        $this->getLogger()->debug($this->indent . 'Teams::purge() start.');

        if ($this->get($team->id)) {
            $update = new Delete($this->getTableGateway()->getTable());
            $update->where([$this->columns['id'] => $team->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ' . $team->id . ' has been purged, as in permanently deleted.');
            $this->getLogger()->debug($this->indent . 'Teams::purge() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Team ID ' . $team->id . ' does not exists. Purge failed.');
            $this->getLogger()->debug($this->indent . 'Teams::purge() end.');

            throw new \Exception('Team ID does not exist');
        }

    }

}
