<?php

namespace TdySports\Model;

use TdyCommons\Filter\Word\LowercaseFirst;
use TdyCommons\Model\Model;
use TdySports\Table\Leagues;
use TdySports\Table\Teams;
use Zend\Filter\FilterChain;
use Zend\Filter\Word\DashToCamelCase;
use Zend\Filter\Word\UnderscoreToCamelCase;
use Zend\ServiceManager\ServiceLocatorInterface;
use Zend\ServiceManager\ServiceManager;

/**
 * Class Standing
 *
 * @package TdySports\Model
 *
 * @author  James Lloyd Atwil <james@nadows.com>
 */
class Standing extends Model
{

    /**
     * The unique ID for Sport.
     *
     * @var int
     */
    public $id = 0;

    /**
     * @var int
     */
    public $teamId = 0;

    /**
     * @var int
     */
    public $leagueId = 0;

    /**
     * @var int
     */
    public $seasonId = 0;

    /**
     * @var int
     */
    public $type = 0;

    /**
     * @var int
     */
    public $wins = 0;

    /**
     * @var int
     */
    public $losses = 0;

    /**
     * @var int
     */
    public $ties = 0;

    /**
     * @var float
     */
    public $percentage = 0.0000;


    /**
     * @var null|ServiceManager|ServiceLocatorInterface
     */
    protected $sm;

    /**
     * @param null $data
     * @param null $sm
     */
    public function __construct($data = null, $sm = null)
    {
        if (!empty($data) && is_array($data)) {
            $this->exchangeArray($data);
        }

        $this->sm = $sm;
    }

    /**
     * @param null|ServiceManager|ServiceLocatorInterface $sm
     */
    public function setServiceLocator($sm)
    {
        $this->sm = $sm;
    }

    /**
     * @see Model::exchangeArray()
     *
     * @param array $data
     */
    public function exchangeArray($data)
    {
        parent::exchangeArray($data);

        foreach ($data as $key => $value) {
            $filter = new FilterChain();
            if (strpos($key, '-') !== false) {
                $filter->attach(new DashToCamelCase());
            } else {
                $filter->attach(new UnderscoreToCamelCase());
            }

            $filter->attach(new LowercaseFirst());
            $nKey = $filter->filter($key);

            if (!property_exists(__CLASS__, $nKey)) {
                continue;
            } else {
                $inArray = in_array($nKey, ['id', 'teamId', 'leagueId', 'wins', 'losses', 'ties']);
                if ($inArray) {
                    $this->$nKey = (int) $value;
                } else {
                    $this->$nKey = $value;
                }
            }
        }
    }

    public function calculatePercentage()
    {
        $totalGames = $this->wins + $this->losses + $this->ties;

        if ($totalGames == 0) {
            $this->percentage = 0.0000;

            return $this->percentage;
        }

        $this->percentage = $this->wins / $totalGames;

        return $this->percentage;
    }

    /**
     * @return null|League
     */
    public function getLeague()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Leagues $table
         * @var League  $sport
         */
        $table = $this->sm->get('TdySports\Table\Leagues');
        $sport = $table->get($this->leagueId);

        return $sport;
    }

    /**
     * @return null|Team
     */
    public function getTeam()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Teams $table
         * @var Team  $sport
         */
        $table = $this->sm->get('TdySports\Table\Teams');
        $team  = $table->get($this->teamId);
        $team->setServiceLocator($this->sm);

        return $team;
    }


    /**
     * @return \StdClass
     */
    public function toStdClass()
    {
        $object = parent::toStdClass();

        if (is_null($this->sm)) {
            return $object;
        }

        unset($object->sportId);


        unset($object->teamId);
        $team         = $this->getTeam();
        $object->team = $team->toStdClass();

        unset($object->leagueId);
        $league         = $this->getLeague();
        $object->league = $league->toStdClass();

        return $object;
    }

}
