<?php

namespace TdyAdmins\Table;

use TdyAdmins\Model;
use TdyCommons\Table\Table;
use Zend\Crypt\Password\Bcrypt;
use Zend\Db\ResultSet\HydratingResultSet;
use Zend\Db\Sql\Delete;
use Zend\Db\Sql\Insert;
use Zend\Db\Sql\Predicate;
use Zend\Db\Sql\Select;
use Zend\Db\Sql\Sql;
use Zend\Db\Sql\Update;
use Zend\Db\Sql\Where;
use Zend\Paginator\Adapter\DbSelect;
use Zend\Paginator\Paginator;

/**
 * Class Admins
 *
 * @package TdyAdmins\Table
 *
 * @author  James Lloyd Atwil <james@nadows.com>
 */
class Admins extends Table
{

    /**
     * @param string $q
     * @param array  $exclusion
     * @param array  $filters
     * @param array  $sort
     * @param bool   $paginated
     *
     * @return bool|HydratingResultSet|Paginator
     */
    public function fetch($q = '', $exclusion = [], $filters = [], $sort = [], $paginated = false)
    {
        $this->getLogger()->debug($this->indent . 'Admins::fetch() start.');

        $select = new Select();
        $select->from($this->getTableGateway()->getTable());
        $conditions = [];

        $where = new Where([]);
        $where->addPredicate(new Predicate\Expression('is_active = ?', 1));
        $conditions[] = $where;

        if (!empty($q)) {
            $where = new Where([], Predicate\PredicateSet::OP_OR);
            $where->addPredicate(new Predicate\Like($this->columns['first-name'], '%' . $q . '%'));
            $where->addPredicate(new Predicate\Like($this->columns['last-name'], '%' . $q . '%'));
            $conditions[] = $where;
        }

        if (!empty($filters)) {
            $where = new Where([]);
            foreach ($filters as $k => $v) {
                if (is_array($v)) {
                    $where->addPredicate(new Predicate\In($k, $v));
                } else {
                    $where->addPredicate(new Predicate\Expression($k . ' = ?', $v));
                }
            }
            $conditions[] = $where;
        }

        if (!empty($exclusion)) {
            $where = new Where([]);
            $where->addPredicate(new Predicate\NotIn('id', $exclusion));
            $conditions[] = $where;
        }

        $select->where($conditions, Predicate\PredicateSet::OP_AND);

        if (!is_null($sort)) {
            if ($this->isSortEmpty($sort)) {
                $select->order('last_name ASC, first_name ASC');
            } else {
                $select->order($sort);
            }
        } else {
            $select->order($sort);
        }

        $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $select->getSqlString($this->getAdapter()->getPlatform()));

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\Admin());

        if ($paginated) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: YES');

            $paginatorAdapter = new DbSelect($select, $this->getTableGateway()->getSql(), $resultSet);
            $paginator        = new Paginator($paginatorAdapter);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns Paginator instance.');
            $this->getLogger()->debug($this->indent . 'Admins::fetch() end.');

            return $paginator;
        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: NO');

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($select);

            $results = $statement->execute();
            $resultSet->initialize($results);

            if (!$resultSet) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns FALSE');
                $this->getLogger()->debug($this->indent . 'Admins::fetch() end.');

                return false;
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns HydratingResultSet instance.');
                $this->getLogger()->debug($this->indent . 'Admins::fetch() end.');

                return $resultSet;
            }
        }
    }

    /**
     * Returns the Admin.
     *
     * @param int $id
     *
     * @return Model\Admin|null
     * @throws \Exception
     */
    public function get($id)
    {
        $rowset = $this->tableGateway->select([$this->columns['id'] => $id]);
        $row    = $rowset->current();
        if (!$row) {
            return null;
        }

        return $row;
    }

    /**
     * @param string $name
     * @param int    $exclude
     *
     * @return Model\Admin|bool|null
     */
    public function getByUsername($name, $exclude = 0)
    {
        $rowset = $this->tableGateway->select(function (Select $select) use ($name, $exclude) {
            $select->where(["username = '" . $name . "'"]);
            $select->where(["is_active = 1"]);
            if ($exclude > 0) {
                $select->where(["id != '" . $exclude . "'"]);
            }
        });
        $row    = $rowset->current();
        if (!$row) {
            return false;
        }

        return $row;
    }

    /**
     * @param string $email
     * @param int    $exclude
     *
     * @return Model\Admin|bool|null
     */
    public function getByEmail($email, $exclude = 0)
    {
        $rowset = $this->tableGateway->select(function (Select $select) use ($email, $exclude) {
            $select->where(['email' => $email]);
            $select->where(['is_active' => 1]);
            if ($exclude > 0) {
                $select->where(["id != '" . $exclude . "'"]);
            }
        });
        $row    = $rowset->current();
        if (!$row) {
            return false;
        }

        return $row;
    }

    /**
     * @param Model\Admin $admin
     *
     * @return Model\Admin
     * @throws \Exception
     */
    public function save(Model\Admin $admin)
    {
        $this->getLogger()->debug($this->indent . 'Admins::save() start.');

        $data = [
            $this->columns['username']    => $admin->username,
            $this->columns['last-name']   => $admin->lastName,
            $this->columns['first-name']  => $admin->firstName,
            $this->columns['middle-name'] => $admin->middleName,
            $this->columns['suffix']      => $admin->suffix,
            $this->columns['nickname']    => $admin->nickname,
            $this->columns['email']       => $admin->email,
        ];

        if ($admin->password != '' && !is_null($admin->password)) {
            $bcrypt                           = new Bcrypt();
            $data[$this->columns['password']] = $bcrypt->create($admin->password);
        }

        $id = (int) $admin->id;
        if ($id == 0) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Inserting Admin record.');

            $data['is_active']  = 1;
            $data['created_by'] = $admin->createdBy;
            $data['created_on'] = date('Y-m-d H:i:s');

            $insert = new Insert($this->getTableGateway()->getTable());
            $insert->values($data);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $insert->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($insert);

            $statement->execute();

            $id = $this->getAdapter()->getDriver()->getLastGeneratedValue();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ' . $id . ' has been inserted.');
            $this->getLogger()->debug($this->indent . 'Admins::save() end.');

            return $this->get($id);
        } else {
            if ($this->get($id)) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Updating Admin record.');

                $data[$this->columns['updated-by']] = $admin->updatedBy;
                $data[$this->columns['updated-on']] = date('Y-m-d H:i:s');

                $update = new Update($this->getTableGateway()->getTable());
                $update->set($data);
                $update->where([$this->columns['id'] => $id]);

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

                $sql       = new Sql($this->tableGateway->getAdapter());
                $statement = $sql->prepareStatementForSqlObject($update);

                $statement->execute();

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ' . $id . ' has been updated.');
                $this->getLogger()->debug($this->indent . 'Admins::save() end.');

                return $this->get($id);
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ID ' . $id . ' does not exists. Update failed.');
                $this->getLogger()->debug($this->indent . 'Admins::save() end.');

                throw new \Exception('Admin ID does not exist');
            }
        }
    }

    /**
     * @param Model\Admin $admin
     *
     * @throws \Exception
     */
    public function delete(Model\Admin $admin)
    {
        $this->getLogger()->debug($this->indent . 'Admins::delete() start.');

        if ($this->get($admin->id)) {
            $data = [
                $this->columns['is-active']  => 0,
                $this->columns['deleted-by'] => $admin->deletedBy,
                $this->columns['deleted-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $admin->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ' . $admin->id . ' has been deleted.');
            $this->getLogger()->debug($this->indent . 'Admins::delete() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ID ' . $admin->id . ' does not exists. Deletion failed.');
            $this->getLogger()->debug($this->indent . 'Admins::delete() end.');

            throw new \Exception('Admin ID does not exist');
        }
    }

    /**
     * Restore Admin to the table. Sets the Admin to active.
     *
     * @param Model\Admin $admin
     *
     * @throws \Exception
     */
    public function restore(Model\Admin $admin)
    {
        $this->getLogger()->debug($this->indent . 'Admins::restore() start.');

        if ($this->get($admin->id)) {
            $data = [
                $this->columns['is-active']  => 1,
                $this->columns['updated-by'] => $admin->updatedBy,
                $this->columns['updated-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $admin->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ' . $admin->id . ' has been restored.');
            $this->getLogger()->debug($this->indent . 'Admins::restore() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ID ' . $admin->id . ' does not exists. Restoration failed.');
            $this->getLogger()->debug($this->indent . 'Admins::restore() end.');

            throw new \Exception('Admin ID does not exist');
        }
    }

    /**
     * Set a new password and save to the table.
     *
     * @param Model\Admin $admin
     *
     * @throws \Exception
     */
    public function resetPassword(Model\Admin $admin)
    {
        $this->getLogger()->debug($this->indent . 'Admins::resetPassword() start.');

        if ($this->get($admin->id)) {
            $data = [
                $this->columns['password']   => $admin->password,
                $this->columns['updated-by'] => $admin->updatedBy,
                $this->columns['updated-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $admin->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Password of the Admin ' . $admin->id . ' has been reset.');
            $this->getLogger()->debug($this->indent . 'Admins::resetPassword() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ID ' . $admin->id . ' does not exists. Password reset failed.');
            $this->getLogger()->debug($this->indent . 'Admins::resetPassword() end.');

            throw new \Exception('Admin ID does not exist');
        }
    }

    /**
     * Permanently delete the Admin.
     *
     * @param Model\Admin $admin
     *
     * @throws \Exception
     */
    public function purge(Model\Admin $admin)
    {
        $this->getLogger()->debug($this->indent . 'Admins::purge() start.');

        if ($this->get($admin->id)) {
            $update = new Delete($this->getTableGateway()->getTable());
            $update->where([$this->columns['id'] => $admin->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ' . $admin->id . ' has been purged, as in permanently deleted.');
            $this->getLogger()->debug($this->indent . 'Admins::purge() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Admin ID ' . $admin->id . ' does not exists. Purge failed.');
            $this->getLogger()->debug($this->indent . 'Admins::purge() end.');

            throw new \Exception('Admin ID does not exist');
        }

    }

}
